/* ==================================================
   qt_test_gui project
   Created: Geoff R. McLane - Aug 2011
   License: GPL2 (or later)
   ================================================== */
// load_awy.cpp

#include "test_config.h"
#include "load_awy.h"
#include "utilities/fgx_gzlib.h"
#include "utilities.h"

QString def_fgawy_file(DEF_FGAWY_FILE);
QString def_xpawy_file(DEF_XPAWY_FILE);

QString get_fgawy_dat() { return def_fgawy_file; }
void set_fgawy_dat(QString file) { def_fgawy_file = file; }
QString get_xpawy_dat() { return def_xpawy_file; }
void set_xpawy_dat(QString file) { def_xpawy_file = file; }


/* -----------------------------
   from : http://data.x-plane.com/file_specs/Awy640.htm
ABCDE       Name of intersection or nav-aid at the beginning of this segment (the fix ABCDE in this example).
32.283733   Latitude of the beginning of this segment.
-106.898669 Longitude of the beginning of this segment.
ABC         Name of intersection or nav-aid at the end of this segment (the nav-aid ABC in this example).
33.282503   Latitude of the end of this segment.
-107.280542 Longitude of the end of this segment.
2           This is a "High" airway (1 = "low", 2 = "high").  If an airway segment is both High and Low, then it should be listed twice (once in each category).  This determines if the airway is shown on X-Plane's "High Enroute" or "Low Enroute" charts.
180         Base of airway in hundreds of feet (18,000 feet in this example).
450         Top of airways in hundreds of feet (45,000 feet in this example).
J13         Airway segment name.  If multiple airways share this segment, then all names will be included separated by a hyphen (eg. "J13-J14-J15")
   ----------------------------- */

enum AwyOff {
    AW_NAME = 0, // ABCDE       Name of intersection or nav-aid at the beginning of this segment (the fix ABCDE in this example).
    AW_BLAT = 1, // 32.283733   Latitude of the beginning of this segment.
    AW_BLON = 2, // -106.898669 Longitude of the beginning of this segment.
    AW_INTER = 3,// ABC         Name of intersection or nav-aid at the end of this segment (the nav-aid ABC in this example).
    AW_ELAT = 4, // 33.282503   Latitude of the end of this segment.
    AW_ELON = 5, // -107.280542 Longitude of the end of this segment.
    AW_HILO = 6, // 2           This is a "High" airway (1 = "low", 2 = "high").  If an airway segment is both High and Low, then it should be listed twice (once in each category).  This determines if the airway is shown on X-Plane's "High Enroute" or "Low Enroute" charts.
    AW_BALT = 7, // 180         Base of airway in hundreds of feet (18,000 feet in this example).
    AW_TALT = 8, // 450         Top of airways in hundreds of feet (45,000 feet in this example).
    AW_SNAM = 9, // J13         Airway segment name.  If multiple airways share this segment, then all names will be included separated by a hyphen (eg. "J13-J14-J15")
    AW_SIZE = 10
};

typedef struct tagAWYDAT {
    QString name;
    double blat, blon;
    QString inter;
    double elat, elon;
    int hilo;
    int balt, talt;
    QString segname;
    int flag;
}AWYDAT, * PAWYDAT;

typedef QList<PAWYDAT> AWY_LIST;
typedef AWY_LIST * PAWY_LIST;

AWY_LIST fgawy_list;
AWY_LIST xpawy_list;
QStringList fgawy_lines, xpawy_lines;

void clear_awy_lists()
{
    fgawy_lines.clear();
    xpawy_lines.clear();
    PAWYDAT pad;
    while (!fgawy_list.isEmpty()) {
        pad = fgawy_list.takeFirst();
        delete pad;
    }
    while (!xpawy_list.isEmpty()) {
        pad = xpawy_list.takeFirst();
        delete pad;
    }
}

int load_awy_file(QString file, QStringList & sl, PAWY_LIST pal)
{
    QTime tm;
    QFile f(file);
    if (! f.exists() ) {
        outLog("ERROR: Can NOT locate file ["+file+"]");
        return 1;
    }
    fgx_gzHandle gzf; // opaque file gz handle
    tm.start();
    gzf = fgx_gzOpen(file);
    if (!gzf) {
        outLog("ERROR: Failed to open ["+file+"]");
        return 2;
    }
    outLog("Processing file ["+file+"]");

    QString end("99");
    QString name;
    //* ignore first line
    fgx_gzReadline(gzf);
    QString credits = fgx_gzReadline(gzf);

    int version = 999;
    if(credits.startsWith("640 Version")) {
        version = 640;
        outLog("Dealing with Version 640");
    } else {
        credits.chop(credits.length() - 12);
        outLog("Dealing with "+credits);
    }

    QString line;
    QStringList parts;
    int line_counter = 2;
    int pcnt;
    while ( ! fgx_gzEof(gzf) ) {
        line = fgx_gzReadline(gzf);
        line_counter++;
        line = line.trimmed();
        if (line.length() == 0)
            continue;
        parts = line.split(" ", QString::SkipEmptyParts);
        pcnt = parts.size();
        name = parts[AW_NAME]; // ABCDE       Name of intersection or nav-aid at the beginning of this segment (the fix ABCDE in this example).
        if (name == end)
            break;  // last line is "99"
        if (pcnt < AW_SIZE) {
            outLog("Discarding line ["+line+"]");
            continue;
        }
        PAWYDAT pad = new AWYDAT;
        pad->name = name; // ABCDE       Name of intersection or nav-aid at the beginning of this segment (the fix ABCDE in this example).
        pad->blat = parts[AW_BLAT].toDouble(); // 32.283733   Latitude of the beginning of this segment.
        pad->blon = parts[AW_BLON].toDouble(); // -106.898669 Longitude of the beginning of this segment.
        pad->inter = parts[AW_INTER]; // ABC         Name of intersection or nav-aid at the end of this segment (the nav-aid ABC in this example).
        pad->elat = parts[AW_ELAT].toDouble(); // 33.282503   Latitude of the end of this segment.
        pad->elon = parts[AW_ELON].toDouble(); // -107.280542 Longitude of the end of this segment.
        pad->hilo = parts[AW_HILO].toInt(); // 2           This is a "High" airway (1 = "low", 2 = "high").  If an airway segment is both High and Low, then it should be listed twice (once in each category).  This determines if the airway is shown on X-Plane's "High Enroute" or "Low Enroute" charts.
        pad->balt = parts[AW_BALT].toInt(); // 180         Base of airway in hundreds of feet (18,000 feet in this example).
        pad->talt = parts[AW_TALT].toInt(); // 450         Top of airways in hundreds of feet (45,000 feet in this example).
        pad->segname = parts[AW_SNAM]; // J13         Airway segment name.  If multiple airways share this segment, then all names will be included separated by a hyphen (eg. "J13-J14-J15")
        // AW_SIZE = 10
        pal->append(pad);
        sl.append(line);
    }
    fgx_gzClose(gzf);
    QString msg;
    msg.sprintf("Done %d lines, found %d items...", line_counter,sl.count());
    int ms = tm.elapsed();
    outLog(msg+", in "+getElapTimeStg(ms));

    return 0;
}

QString get_awy_string( PAWYDAT pad )
{
    QString msg, posn;
    msg = pad->name; // ABCDE       Name of intersection or nav-aid at the beginning of this segment (the fix ABCDE in this example).
    posn.sprintf(" %f %f ", pad->blat, pad->blon );
    msg.append(posn);
    msg.append(pad->inter); // ABC         Name of intersection or nav-aid at the end of this segment (the nav-aid ABC in this example).
    posn.sprintf(" %f %f ", pad->elat, pad->elon );
    msg.append(posn);
    posn.sprintf(" %d %d %d ", pad->hilo, pad->balt, pad->talt );
    msg.append(posn);
    msg.append(pad->segname); // J13         Airway segment name.  If multiple airways share this segment, then all names will be included separated by a hyphen (eg. "J13-J14-J15")
    return msg;
}

void cmp_awy_files(QString file1, QStringList & sl1, PAWY_LIST pal1,
                   QString file2, QStringList & sl2, PAWY_LIST pal2)
{
    QString msg;
    int cnt1 = pal1->count();
    int cnt2 = pal2->count();
    msg.sprintf("Comp FG %d, %d lines",cnt1, sl1.count());
    msg.append(", file ["+file1+"]");
    outLog(msg);
    msg.sprintf("With XP %d, %d lines",cnt2, sl2.count());
    msg.append(", file ["+file2+"]");
    outLog(msg);
    //AWYDAT awd1, awd2;
    PAWYDAT pad1, pad2;
    int i, j, fnd;
    double blat1, blon1, elat1, elon1;
    double blat2, blon2, elat2, elon2;
    QString name1, name2;
    int not_found1 = 0;
    int not_found2 = 0;
    double dist, min_dist;
    int min_off;
    for (i = 0; i < cnt1; i++) {
        pad1 = pal1->at(i);
        pad1->flag = 0;
    }
    for (j = 0; j < cnt2; j++) {
        pad2 = pal2->at(j);
        pad2->flag = 0;
    }
    msg.sprintf("Comparing FG %d, with XP %d", cnt1, cnt2);
    outLog(msg);
    for (i = 0; i < cnt1; i++) {
        pad1 = pal1->at(i);
        name1 = pad1->name;
        blat1 = pad1->blat;
        blon1 = pad1->blon;
        elat1 = pad1->elat;
        elon1 = pad1->elon;
        fnd = 0;
        min_off = -1;
        min_dist = 9999999.9;
        for (j = 0; j < cnt2; j++) {
            pad2 = pal2->at(j);
            if (pad2->flag)
                continue; // already matched
            name2 = pad2->name;
            blat2 = pad2->blat;
            blon2 = pad2->blon;
            elat2 = pad2->elat;
            elon2 = pad2->elon;
            if (name1 == name2) {
                fnd = 1;
                pad1->flag = j + 1;
                pad2->flag = i + 1;
                break; // got a name match
            }
            dist = dist_est_km(blat1, blon1, blat2, blon2);
            if (dist < min_dist) {
                min_dist = dist;
                min_off = j;
            }
        }
        if (fnd) {
            // could check MORE
        } else {
            // NOT FOUND
            not_found1++;
            msg.sprintf("FG:%d:%d: ", not_found1, (i+1));
            msg.append(get_awy_string(pad1)+" NF in XP");
            outLog(msg);
            if (min_off >= 0) {
                msg.sprintf("Nearest XP at %.1f Km ",min_dist);
                pad2 = pal2->at(min_off);
                msg.append(get_awy_string(pad2)+"!");
                outLog(msg);
            }
        }
    }
    msg.sprintf("Comparing XP %d, with FG %d", cnt2, cnt1);
    outLog(msg);
    for (j = 0; j < cnt2; j++) {
        pad2 = pal2->at(j);
        if (pad2->flag)
            continue;
        name2 = pad2->name;
        blat2 = pad2->blat;
        blon2 = pad2->blon;
        elat2 = pad2->elat;
        elon2 = pad2->elon;
        fnd = 0;
        min_off = -1;
        min_dist = 9999999999.0;
        for (i = 0; i < cnt1; i++) {
            pad1 = pal1->at(i);
            if (pad1->flag)
                continue;
            name1 = pad1->name;
            blat1 = pad1->blat;
            blon1 = pad1->blon;
            elat1 = pad1->elat;
            elon1 = pad1->elon;
            if (name1 == name2) {
                fnd = 1;
                pad1->flag = j + 1;
                pad2->flag = i + 1;
                break;
            }
            dist = dist_est_km(blat1, blon1, blat2, blon2);
            if (dist < min_dist) {
                min_dist = dist;
                min_off = i;
            }
        }
        if (fnd) {
            // could check MORE
        } else {
            // not found
            not_found2++;
            msg.sprintf("XP:%d:%d: ", not_found2, (j+1));
            msg.append(get_awy_string(pad2)+" NF in FG");
            outLog(msg);
            if (min_off >= 0) {
                msg.sprintf("Nearest FG at %.1f Km ",min_dist);
                pad1 = pal1->at(min_off);
                msg.append(get_awy_string(pad1)+"!");
                outLog(msg);
            }
        }
    }
    int same1 = (cnt1 - not_found1);
    double pct1 = (double)same1 / (double)cnt1;
    int ipct1 = (int)((pct1 + 0.0005) * 1000.0);
    pct1 = (double)ipct1 / 10.0;
    double pct2 = (double)(cnt2 - not_found2) / (double)cnt2;
    int ipct2 = (int)((pct2 + 0.0005) * 1000.0);
    pct2 = (double)ipct2 / 10.0;
    msg.sprintf("AWY: FG NF %5d of %6d, %5.1f%% SAME, XP NF %5d of %6d, %5.1f%% SAME %6d",
                not_found1, cnt1, pct1,
                not_found2, cnt2, pct2, same1);
    outLog(msg,0x4001);
}

void load_awy_files()
{
    clear_awy_lists();
    QString def_fgawy = get_fgawy_dat();
    QString def_xpawy = get_xpawy_dat();
    load_awy_file(def_fgawy,fgawy_lines,&fgawy_list);
    load_awy_file(def_xpawy,xpawy_lines,&xpawy_list);
    cmp_awy_files(def_fgawy,fgawy_lines,&fgawy_list,
                  def_xpawy,xpawy_lines,&xpawy_list);

}

// eof - load_awy.cpp
