// load_apt810.cpp

#include "test_config.h"
#include "mainwindow.h"
#include "test_aptdat.h"
#include "utilities/fgx_gzlib.h"
#include "utilities/simgear/simgear.h"
#include "utilities.h"

bool import_icao_only = false;

/* ------------------------------------
   from : http://data.x-plane.com/file_specs/Apt810.htm
Code (apt.dat) Used for
1 Airport header data.
16 Seaplane base header data. No airport buildings or boundary fences will be rendered in X-Plane.
17 Heliport header data.  No airport buildings or boundary fences will be rendered in X-Plane.
10 Runway or taxiway at an airport.
14 Tower view location.
15 Ramp startup position(s)
18 Airport light beacons (usually "rotating beacons" in the USA).  Different colours may be defined.
19 Airport windsocks.
50 to 56 Airport ATC (Air Traffic Control) frequencies.

"10  52.908983  156.853257 xxx 179.98   546 0.0 0.0   124 111111  2 0 0 0.00 0";

Runways and taxiways Example Usage
 0: 10          Identifies this as a data line for a runway or taxiway segment.
 1: 35.044209   Latitude (in decimal degrees) of runway or taxiway segment center.
 2: -106.598557 Longitude (in decimal degrees) of runway or taxiway segment center.
 3: 08x         Runway number (eg 25x or 24R).  If there is no runway suffix (eg. L, R, C or "S"), then an x is used.  xxx identifies the entry as a taxiway.  Helipads at the same airport are numbered sequentially as "H1x", H2x".
 4: 90.439      True (not magnetic) heading of the runway in degrees.  Must be between 0.00 and 360.00.
 5: 13749       Runway or taxiway segment length in feet.

 0: 10          Identifies this as a data line for a runway or taxiway segment.
 1: 35.044209   Latitude (in decimal degrees) of runway or taxiway segment center.
 2: -106.598557 Longitude (in decimal degrees) of runway or taxiway segment center.
 3: 08x         Runway number (eg 25x or 24R).  If there is no runway suffix (eg. L, R, C or "S"), then an x is used.  xxx identifies the entry as a taxiway.  Helipads at the same airport are numbered sequentially as "H1x", H2x".
 4: 90.439      True (not magnetic) heading of the runway in degrees.  Must be between 0.00 and 360.00.
 5: 13749       Runway or taxiway segment length in feet.
 6: 1000.0000   Length of displaced threshold (1,000 feet) for runway 08 and for the reciprocal runway 26 (0 feet).  The length of the reciprocal runways displaced threshold is expressed as the fractional part of this number.  Take the runway 26 displaced threshold length  (in feet) and divide it by 10,000, then add it to the displaced threshold length for runway 08.  For example, for displaced threshold lengths of 543 feet and 1234 feet, the code would be 543.1234.
            Note that the displaced threshold length is included in the overall runway length but that the stopway length is excluded from the overall runway length.  This code should be 0.0000 for taxiway segments.  FYI, the displaced threshold is usually marked (in the real world) with long white arrows pointing toward the threshold.  The displaced threshold is not available for use by aeroplanes landing, but may be used for take-off (in practice, if you use these last few feet of the runway for take-off, you are probably in serious trouble!).
 7: 0.1000      Length of stopway/blastpad/over-run at the approach end of runway 08 (0 feet) and for runway 26 (1,000 feet), using the same coding structure defined above.  FYI, in the real world the stopway/blastpad/over-run is usually marked with large yellow chevrons, and aeroplane movements are not permitted.
 8: 150         Runway or taxiway segment width in feet.
 9: 252231      Runway or taxiway segment lighting codes.
            The first three digits ("252") define the lighting for the runway as seen when approached from the direction implied by the runway number (08 in our example).
            The final three ("231") define the lighting for the runway as seen when approached from the opposite end (26 in our example).
            In order, these codes represent:
              Runway end A (08): Visual approach path (VASI / PAPI etc.) lighting.  Here, code 2 corresponds to a VASI.
              Runway end A (08): Runway lighting. Here, code 5 corresponds to TDZ lighting, which also implies centre-line lighting, REIL and edge lighting.
              Runway end A (08): Approach lighting.  Here, code 2 corresponds to SSALS.
              Other runway end (26): Visual approach path (VASI / PAPI etc.) lighting. Here, code 2 corresponds to a VASI.
              Other runway end (26): Runway lighting. Here, code 3 corresponds to REIL, which also implies edge lighting.
              Other runway end (26): Approach lighting. Here, code 1 implies no approach lighting.
10: 02           Runway or taxiway surface code for the runway or taxiway segment.  The leading zero is optional - but I always use it to keep all the columns neatly lined up.
11: 0            Runway shoulder code.  These are only available in file version 701 and later.  Here, code 0 implies that there is no runway shoulder.
12: 3            Runway markings (the white painted markings on the surface of the runway.  Here, code 3 implies precision runway markings (ie. there is an associated precision approach for the runway, either an ILS or MLS).
13: 0.25         Runway smoothness.  Used to cause bumps when taxying or rolling along the runway in X-Plane.  It is on a scale of 0.0 to 1.0, with 0.0 being very smooth, and 1.0 being very, very rough.  X-Plane determines a baseline smoothness based upon the runway surface type, and then uses this factor to determine the 'quality' of the runway surface.  The default value is 0.25.
14: 1            Runway has 'distance remaining' signs (0=no signs, 1=show signs).  These are the white letters on a black background on little illuminated signs along a runway, indicating the number of thousands of feet of usable runway that remain.  They are inappropriate at small airports or on most dirt, gravel or grass runways.
15: 0300.0350    NEW for file version 810:  Visual glideslope angle for the VASI or PAPI at each end of the runway (3.00 degrees for runway 08 and 3.50 degrees for runway 26).  The angle for runway 08 is the whole part of this number divided by 100 (so "0300" becomes 3.00 degrees) and the angle for the reciprocal runway (26) is the fractional part of this number multiplied by 100 (so "0.0350" becomes 3.50 degrees).  This data is required for runways, but is NOT necessary for taxiways.
   ------------------------------------ */

#define a81_Airport   1  // Airport header data.
#define a81_Seaplane 16  // Seaplane base header data. No airport buildings or boundary fences will be rendered in X-Plane.
#define a81_Heliport 17  // Heliport header data.  No airport buildings or boundary fences will be rendered in X-Plane.
#define a81_Runway   10  // Runway or taxiway at an airport.
#define a81_Tower    14  // Tower view location.
#define a81_Ramp     15  // Ramp startup position(s)
#define a81_Lights   18  // Airport light beacons (usually "rotating beacons" in the USA).  Different colours may be defined.
#define a81_Windsock 19  // Airport windsocks.
// 19   36.97040984  127.02064840 1 WS
// 19   36.95401518  127.04149243 1 WS
enum ws810 {
    WS81_CODE = 0,
    WS81_LAT = 1,
    WS81_LON = 2,
    WS81_LIT = 3, // Flag for windsock lighting 0=unlit, 1=illuminated
    WS81_NAME = 4, // OPTIONAL
    WS81_MIN = 5
};

#define a81_Com_min  50
#define a81_Com_max  56  // Airport ATC (Air Traffic Control) frequencies.
// 50 10820 ATIS
// 53 11950 DESIDERIO GND
// 54 12250 DESIDERIO TWR
// 55 12790 OSAN APP/DEP
enum Comm810 {
    C81_CODE = 0,
    C81_FREQ = 1,
    C81_NAME = 2,
    C81_MIN = 3
};


// Runways and taxiways Example Usage
enum RwyOff {
    RW_CODE = 0, //10 Identifies this as a data line for a runway or taxiway segment.
    RW_LAT = 1,  //35.044209   Latitude (in decimal degrees) of runway or taxiway segment center.
    RW_LON = 2,  //-106.598557 Longitude (in decimal degrees) of runway or taxiway segment center.
    RW_NUM = 3,  //08x Runway number (eg 25x or 24R).  If there is no runway suffix (eg. L, R, C or "S"), then an x is used.  xxx identifies the entry as a taxiway.  Helipads at the same airport are numbered sequentially as "H1x", H2x".
    RW_HEAD = 4, //90.439 True (not magnetic) heading of the runway in degrees.  Must be between 0.00 and 360.00.
    RW_LEN = 5,  //13749 Runway or taxiway segment length in feet.
    RW_DISP = 6, //1000.0000   Length of displaced threshold (1,000 feet) for runway 08 and for the reciprocal runway 26 (0 feet).  The length of the reciprocal runways displaced threshold is expressed as the fractional part of this number.  Take the runway 26 displaced threshold length  (in feet) and divide it by 10,000, then add it to the displaced threshold length for runway 08.  For example, for displaced threshold lengths of 543 feet and 1234 feet, the code would be 543.1234.
    //  Note that the displaced threshold length is included in the overall runway length but that the stopway length is excluded from the overall runway length.  This code should be 0.0000 for taxiway segments.  FYI, the displaced threshold is usually marked (in the real world) with long white arrows pointing toward the threshold.  The displaced threshold is not available for use by aeroplanes landing, but may be used for take-off (in practice, if you use these last few feet of the runway for take-off, you are probably in serious trouble!).
    RW_STOP = 7, // 0.1000  Length of stopway/blastpad/over-run at the approach end of runway 08 (0 feet) and for runway 26 (1,000 feet), using the same coding structure defined above.  FYI, in the real world the stopway/blastpad/over-run is usually marked with large yellow chevrons, and aeroplane movements are not permitted.
    RW_WID = 8,  // 150     Runway or taxiway segment width in feet.
    RW_LITES = 9,// 252231  Runway or taxiway segment lighting codes.
    RW_SURF = 10,// 02      Runway or taxiway surface code for the runway or taxiway segment.  The leading zero is optional - but I always use it to keep all the columns neatly lined up.
    RW_SHLDR = 11, // 0  Runway shoulder code.  These are only available in file version 701 and later.  Here, code 0 implies that there is no runway shoulder.
    RW_MARKS = 12, // 3  Runway markings (the white painted markings on the surface of the runway.  Here, code 3 implies precision runway markings (ie. there is an associated precision approach for the runway, either an ILS or MLS).
    RW_SMOOTH = 13,// 0.25 Runway smoothness.  Used to cause bumps when taxying or rolling along the runway in X-Plane.  It is on a scale of 0.0 to 1.0, with 0.0 being very smooth, and 1.0 being very, very rough.  X-Plane determines a baseline smoothness based upon the runway surface type, and then uses this factor to determine the 'quality' of the runway surface.  The default value is 0.25.
    RW_SIGNS = 14, // 1 Runway has 'distance remaining' signs (0=no signs, 1=show signs).  These are the white letters on a black background on little illuminated signs along a runway, indicating the number of thousands of feet of usable runway that remain.  They are inappropriate at small airports or on most dirt, gravel or grass runways.
    // RW_GS = 15, // 0300.0350 NEW for file version 810:  Visual glideslope angle for the VASI or PAPI at each end of the runway (3.00 degrees for runway 08 and 3.50 degrees for runway 26).  The angle for runway 08 is the whole part of this number divided by 100 (so "0300" becomes 3.00 degrees) and the angle for the reciprocal runway (26) is the fractional part of this number multiplied by 100 (so "0.0350" becomes 3.50 degrees).  This data is required for runways,
    // but is NOT necessary for taxiways.
    RW_SIZE = 15
};

bool load_apt810()
{
    QString msg;
    QTime tm;
    int   ms;

    //QString zf = mainObject->X->fgroot("/Airports/apt.dat.gz");
    QString zf = get_fgapt_dat(); // "C:\\FGCVS\\flightgear\\data\\Airports\\apt.dat.gz";
    fgx_gzHandle gzf; // opaque file gz handle
    QFile f;
    if (!f.exists(zf)) {
        outLog("ERROR: Failed to find ["+zf+"! NO AIRPORT FILE DATA!");
        return true;
    }
    tm.start();
    gzf = fgx_gzOpen(zf);
    if (!gzf) {
        outLog("ERROR: Failed to open ["+zf+"]");
        return true;
    }
    outLog("Processing file ["+zf+"]");

    int air_count = 0;
    int line_counter = 0;
    int rwy_count = 0;
    QRegExp rxICAOAirport("[A-Z]{4}");

    //* ignore first line
    fgx_gzReadline(gzf);
    QString credits = fgx_gzReadline(gzf);

    int version = 999;
    if(credits.startsWith("810 Version")) {
        version = 810;
        outLog("Dealing with Version 810");
    } else {
        credits.chop(credits.length() - 12);
        outLog("Dealing with "+credits);
    }

    QString airport_code;
    QString airport_name;
    QString elevation;
    QString sp(" ");
    //QString apt("1");
    //QString rwy("10");
    QString one("1");
    QString txwy("xxx");
    //QString end("99");
    PAIRPORT2 pa2 = 0;

    QString rwy_number, tmp;
    int elev_ft, tower, bldgs, pcnt, rwy_cnt, p, code;
    double lat,lon,sum_lat,sum_lon, got_apt;
    line_counter = 2; // done 2 lines
    rwy_cnt = 0;
    got_apt = 0;
    elev_ft = 0;
    sum_lat = 0.0;
    sum_lon = 0.0;
    int ws_count = 0;
    int com_count = 0;
    int tx_count = 0;
    clear_apt_list(); // clear any PREVIOUS
    while ( ! fgx_gzEof(gzf) ) {
        QString line = fgx_gzReadline(gzf);
        line_counter++;
        line = line.trimmed();
        if (line.isEmpty())
            continue;
        QString row_code = line.section(' ',0, 0);
        code = row_code.toInt();

        if (code == 99)
            break; // last line

        QStringList parts = line.split(" ", QString::SkipEmptyParts);
        pcnt = parts.size();
        //**********************************************************************
        //*** Airport
        //if(row_code == "1"){
        if ((code == a81_Airport)||(code == a81_Seaplane)||(code == a81_Heliport)) {
            if (rwy_cnt && got_apt) {
                // get 'average' center of last airport
                lat = sum_lat / (double)rwy_cnt;
                lon = sum_lon / (double)rwy_cnt;
                pa2->lat  = lat;
                pa2->lon  = lon;
            }

            got_apt = 0;
            pa2 = 0;
            lat = -200.0;
            lon = -200.0;
            if (pcnt >= 6) {
                // http://data.x-plane.com/file_specs/Apt715.htm and 815
                // 0 = airport code = '10'
                // 1 = elevation (feet)
                // 2 = has tower
                // 3 = has building
                // 4 = ICAO code
                // 5+ description
                elevation = parts[1];
                elev_ft = elevation.toInt();
                tower = (parts[2] == one) ? 1 : 0;
                bldgs = (parts[3] == one) ? 1 : 0;
                airport_code = parts[4];
                airport_name = parts[5];
                for(p = 6; p < pcnt; p++){ //** TODO WTF ?
                    airport_name.append(sp+parts[p]);
                }
                airport_name = airport_name.trimmed();
                if (import_icao_only) {
                    if( rxICAOAirport.exactMatch(airport_code) ) {
                        //airports[airport_code] = airport_name;
                        air_count++;
                        got_apt = 1;
                    }
                } else {
                    //airports[airport_code] = airport_name;
                    air_count++;
                    got_apt = 1;
                } /* if(is_icao) */
            } else {
                // WHAT IS THIS???
                outLog("Discard apt: "+line);
            }
            if (got_apt) {
                pa2 = new AIRPORT2;
                pa2->icao = airport_code;
                pa2->name = airport_name;
                pa2->elev = elev_ft;
                pa2->lat  = lat;
                pa2->lon  = lon;
                apt_list.append(pa2);
            }
            // done airport init
            sum_lat = 0.0;
            sum_lon = 0.0;
            rwy_cnt = 0;
            /* if(row_code == "1") airport */
            //else if (row_code == rwy) {
        } else if (code == 10) {
            // RUNWAY OR TAXIWAY
            // =================
            if ( !got_apt || !pa2 )
                continue;
            if (pcnt < RW_SIZE) {
                outLog("WARNING: skipped RWY ["+line+"]");
                continue;
            }
            /* -----------------
               0  runway or taxiway segment = '10'
               1  Latitude (in decimal degrees)
               2  Longitude (in decimal degrees)
               3  Runway number (eg 25x or 24R).
               //  If there is no runway suffix (eg. L, R, C or "S"),
               //  then an x is used.  xxx identifies the entry as a taxiway.
               //  Helipads at the same airport are numbered sequentially as "H1x", H2x".
               4   Heading True (not magnetic) in degrees
               5   Length Runway or taxiway segment in feet.
               ----------------- */
            rwy_number = parts[RW_NUM];
            if ((rwy_number != txwy) && got_apt) {

                lat = parts[RW_LAT].toDouble();
                lon = parts[RW_LON].toDouble();

                PRUNWAY2 pr2 = new RUNWAY2;
                double lat1, lon1, lat2, lon2, heading, az1, az2, s;
                heading = parts[RW_HEAD].toDouble();
                s = (double)parts[RW_LEN].toInt(); // len, in feet
                //int res =
                _geo_direct_wgs_84 ( lat, lon, heading,
                                     s * SG_FEET_TO_METER,
                                     &lat2, &lon2, &az2 );
                _geo_direct_wgs_84 ( lat, lon, az2,
                                     s * SG_FEET_TO_METER,
                                     &lat1, &lon1, &az1 );
                pr2->lat1 = lat1;
                pr2->lon1 = lon1;
                pr2->lat2 = lat2;
                pr2->lon2 = lon2;
                pr2->i_heading = az1;
                pr2->i_len_ft = (int)s;
                pr2->rwy_no = rwy_number;

                //rwy_list.append(pr2);
                pa2->rways.append(pr2);
                rwy_cnt++;
                // accumulate
                sum_lat += lat;
                sum_lon += lon;
                rwy_count++;
            } else {
                // taxiway
                tx_count++;
            }
        } else {
            if (!got_apt || !pa2)
                continue;
            if (code == a81_Tower) { // 14  // Tower view location.
                //
            } else if (code == a81_Ramp) { // 15  // Ramp startup position(s)
                //
            } else if (code == a81_Lights) { // 18  // Airport light beacons (usually "rotating beacons" in the USA).  Different colours may be defined.
                //
            } else if (code == a81_Windsock) { // 19  // Airport windsocks.
                if (pcnt >= (WS81_MIN - 1)) {
                    PWINDSOCK pws = new WINDSOCK;
                    // 19   36.97040984  127.02064840 1 WS
                    // 19   36.95401518  127.04149243 1 WS
                    pws->code = code;
                    pws->lat = parts[WS81_LAT].toDouble();
                    pws->lon = parts[WS81_LON].toDouble();
                    pws->lit = parts[WS81_LIT].toInt(); // Flag for windsock lighting 0=unlit, 1=illuminated
                    if (pcnt >= WS81_MIN) {
                        pws->name = parts[WS81_NAME];
                    }
                    // WS81_MIN = 5
                    pa2->ws.append(pws);
                } else {
                    outLog("WARNING: skipped WS ["+line+"]");
                }
            } else if ((code >= a81_Com_min)&&(code <= a81_Com_max)) { // 50-56  // Airport ATC (Air Traffic Control) frequencies.
                // Airport communication frequency
                // 50 10820 ATIS
                // 53 11950 DESIDERIO GND
                // 54 12250 DESIDERIO TWR
                // 55 12790 OSAN APP/DEP
                if (pcnt < C81_MIN) {
                    outLog("WARNING: Discarding COM line ["+line+"]");
                    continue;
                }
                PCOMITEM pci = new COMITEM;
                pci->code = code;
                pci->freq = parts[C81_FREQ].toDouble();
                pci->name = parts[C81_NAME];
                for (p = C81_MIN; p < pcnt; p++) {
                    pci->name.append(sp+parts[p]);
                }
                pa2->cl.append(pci);
            } else {
                tmp.sprintf("WARNING:Ln %d:",line_counter);
                outLog(tmp+" Uncased line ["+line+"]");
            }
        }
    } /* end while readline */
    // get any LAST runway list
    if (rwy_cnt && got_apt) {
        // get 'average' center of last airport
        lat = sum_lat / (double)rwy_cnt;
        lon = sum_lon / (double)rwy_cnt;
        pa2->lat  = lat;
        pa2->lon  = lon;
    }
    fgx_gzClose(gzf);

    msg.sprintf("Done %d lines, found %d airports (%d), %d runways, %d taxi, %d ws, %d com",
                line_counter,air_count,
                apt_list.count(),
                rwy_count, tx_count,
                ws_count, com_count);
    ms = tm.elapsed();
    outLog(msg+", in "+getElapTimeStg(ms));

    return false; // no problems
}

// load_apt810.cpp

