/* ===================================================================
   Module: winport.c
   Purpose: To implement opendir(), readdir(), closedir() in WIN32

   And provide some other functions, some for debug
   =================================================================== */

#if HAVE_CONFIG_H
# include <config.h>
#endif
#include <Windows.h>
#include <stdio.h>      /* for prinf() */
#include "winport.h"

#define CHKMEM(a) if(!a) { fprintf( stderr, "ERROR: MEMORY FAILED!\n" ); exit(2); }

/* private (internal) structure - tacked to end of DIR structure */
typedef struct tagWINDIR {
	HANDLE hfind;
	WIN32_FIND_DATA fd;
}WINDIR, * PWINDIR;

#define  VFH(a)   ( a && ( a != INVALID_HANDLE_VALUE ) )

/* some counters - updated in grep.c - output in show_results
   called from closeout.c during atexit function */
size_t  grep_filesfound = 0;
size_t  grep_filessearched = 0;
size_t  grep_dirsfound = 0;
size_t  grep_dirssearched = 0;
size_t  grep_finds = 0;


static int * ptr_errseen = NULL;
static int * ptr_status = NULL;

void set_exit_status( int val )
{
   if(ptr_status)
      *ptr_status = val;
}

/* add trailing '\', if none already */
static void add_trailing_sep( char * path )
{
   size_t len = strlen(path);
   if(len) {
      int c = path[len - 1];
      if( !( (c == '\\') || ( c == '/') ) )
         strcat(path,"\\");
   }
}

PDIR opendir(const char * dir )
{
	PDIR pdir = (PDIR)malloc( sizeof(DIR) + sizeof(WINDIR));
	if( pdir ) {
		char * cp = &pdir->search[0];
      PWINDIR pwd = (PWINDIR)((PDIR)pdir + 1);  /* get 'private' structure */
		strcpy( cp, dir );
      add_trailing_sep(cp);
		strcat( cp, "*.*" ); /* add wild search (for ALL) */
		pwd->hfind = FindFirstFile( cp, &pwd->fd );
		if( VFH(pwd->hfind) ) {
			pdir->is_first = 1;
			return pdir;
		} else {
			free(pdir);
		}
	}
	return NULL;   /* FAILED */
}

PDE readdir (PDIR pdir)
{
	if(pdir) {
      PWINDIR pwd = (PWINDIR)((PDIR)pdir + 1); /* get 'private' structure */
		char * dst = &pdir->de.d_name[0];
		char * src = &pwd->fd.cFileName[0];
		if( pdir->is_first ) {
			pdir->de.d_namlen = (int)strlen(src);
         if ( pwd->fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY )
            pdir->de.d_type = DT_DIR;
         else
            pdir->de.d_type = DT_REG;
			strcpy( dst, src );
			pdir->is_first = 0;
			return &pdir->de;
		} else {
			if( VFH(pwd->hfind) ) {
				if( FindNextFile( pwd->hfind , &pwd->fd ) ) {
					pdir->de.d_namlen = (int)strlen(src);
					strcpy( dst, src );
               if ( pwd->fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY )
                  pdir->de.d_type = DT_DIR;
               else
                  pdir->de.d_type = DT_REG;
					return &pdir->de;
				}
			}
		}
	}
	return NULL;   /* FAILED, or have returned all files */
}

int closedir(PDIR pdir)
{
	if(pdir) {
      PWINDIR pwd = (PWINDIR)((PDIR)pdir + 1);
		if( VFH(pwd->hfind) ) {
			FindClose(pwd->hfind);
		}
		free( pdir );
	}
	return 0;   /* always success! */
}

/* ========== additional functions, some for debug only ========== */
char * get_title_only( char * fname )
{
   char * pname;
   PSPLITTING psp = (PSPLITTING)malloc(sizeof(SPLITTING));
   CHKMEM(psp);
   _splitpath(fname, psp->drive, psp->dir, psp->fname, psp->ext);
   pname = (char *)malloc( strlen(psp->fname) + 1 );
   CHKMEM(pname);
   strcpy(pname, psp->fname);
   free(psp);
   return pname;
}

int has_wild( char * file )
{
   int iret = 0;
   size_t len = strlen(file);
   size_t i;
   for( i = 0; i < len; i++ ) {
      int c = file[i];
      if( ( c == '?' ) || ( c == '*' ) )
         return 1;
   }
   return 0;
}

void UnixTimeToFileTime(time_t t, LPFILETIME pft)
{
  /* Note that LONGLONG is a 64-bit value */
  LONGLONG ll;

  ll = Int32x32To64(t, 10000000) + 116444736000000000;
  pft->dwLowDateTime = (DWORD)ll;
  pft->dwHighDateTime = (DWORD)(ll >> 32);
}

void UnixTimeToSystemTime(time_t t, LPSYSTEMTIME pst)
{
  FILETIME ft;
  UnixTimeToFileTime(t, &ft);
  FileTimeToSystemTime(&ft, pst);
}

/* ===============================================================================
   FILETIME conversion 
   Under UNIX platforms, file times are maintained in the form of a ANSI C
   runtime arithmetic type named 'time_t', which represents seconds since
   midnight January 1, 1970 UTC (coordinated universal time). 

   Under Win32 platforms, file times are maintained primarily in the form
   of a 64-bit FILETIME structure, which represents the number of
   100-nanosecond intervals since January 1, 1601 UTC (coordinate universal time). 

   convert Win32 FILETIME time to UNIX time 
   Uses a void * so other modules do NOT have to 'know' a FILETIME * ...
   =============================================================================== */
void FileTimeToUnixTime( void * vp, time_t * pt)
{
    __int64 const epochOffset = 116444736000000000i64;
    /* Number of 100-nanosecond units between the beginning of the
       Windows epoch (Jan. 1, 1601) and the Unix epoch (Jan. 1, 1970).
    */
    LARGE_INTEGER li;
    __int64        t;
    FILETIME *   pft = (FILETIME *)vp;
    long          ut, nsecs;
    li.LowPart  = pft->dwLowDateTime;
    li.HighPart = pft->dwHighDateTime;
    t  = (li.QuadPart - epochOffset) * 100;  /* nanoseconds */
    ut = (long)(t / 1E9);
    nsecs = (long)(t - ((__int64)ut * 1E9));
    if(pt)
       *pt = ut;
}


/* ===================================================================== */
void show_results( void )
{
#ifdef EXTRA_DEBUG2
    if( ptr_errseen && ptr_status ) {
        int errseen = *ptr_errseen;
        int status = *ptr_status;
        /* this FAILS to return correct status 
        DWORD  dwi;
        if( GetExitCodeProcess( GetCurrentProcess(), &dwi ) )
           status = (int)dwi;
         *********************************** */
        printf( "Exit code: %d"MEOR, (errseen ? 2 : status ));
    }
#endif /* EXTRA_DEBUG2 */

   if( grep_filesfound ) {
      if( grep_filesfound <= grep_filessearched ) {
         printf( "Processed %d files,", grep_filessearched );
      } else {
         printf( "Found %d files, processed %d,",
            grep_filesfound, grep_filessearched );
      }
      if( grep_dirsfound || grep_dirssearched ) {
         if( grep_dirsfound <= grep_dirssearched ) {
            printf( " %d dirs,", grep_dirssearched );
         } else {
            printf( " %d dirs, done %d,", grep_dirsfound, grep_dirssearched );
         }
      }
      printf( " for %d finds ...\n", grep_finds );
   }
}

#ifdef EXTRA_DEBUG2
void show_buffer( char * buf, size_t buflen, char * pname )
{
   size_t ii;
   size_t maxshow = buflen;
   if(maxshow > MX_SHOW_BUF)
      maxshow = MX_SHOW_BUF;
   printf( "%s %d:[", pname, buflen );
   for( ii = 0; ii < maxshow; ii++ )
   {
      int chr = buf[ii];
      if( chr < ' ' ) {
         printf("^");
         chr += '@';
      } else if( chr == '^' ) {
         printf("^");
      }
      printf( "%c", chr );
   }
   printf( "]"MEOR );
}
#endif /* EXTRA_DEBUG2 */

void initialize_win32( int * argcp, char *** argvp, int * esp, int * statp )
{
    int argc = *argcp;
    char ** argv = *argvp;
#ifdef EXTRA_DEBUG2
    int cc;
    //printf( "Running [%s] ...\n", argv[0] );
    printf( "CMDS %d: ", argc - 1 );
    for( cc = 1; cc < argc; cc++ ) {
        printf( "[%s]%d ",
            (argv[cc] ? argv[cc] : "<null>"),
            (argv[cc] ? strlen(argv[cc]) : 0) );
    }
    printf(MEOR);
#endif // #ifdef EXTRA_DEBUG2
    ptr_errseen = esp;
    ptr_status = statp;
    if(ptr_errseen)
        *ptr_errseen = 0;
    if(ptr_status)
        *ptr_status = 0;
}


/* ===================================================================== */

/* =============================
typedef struct _WIN32_FIND_DATA {
   DWORD dwFileAttributes;
   FILETIME ftCreationTime;
   FILETIME ftLastAccessTime;
   FILETIME ftLastWriteTime;
   DWORD nFileSizeHigh;
   DWORD nFileSizeLow;
   DWORD dwReserved0;
   DWORD dwReserved1;
   TCHAR cFileName[MAX_PATH];
   TCHAR cAlternateFileName[14];
} WIN32_FIND_DATA,  *PWIN32_FIND_DATA,  *LPWIN32_FIND_DATA;
This member can be one or more of the following values.

Attribute / Meaning 
FILE_ATTRIBUTE_ARCHIVE
The file or directory is an archive file or directory.
Applications use this attribute to mark files for backup or removal.
 
FILE_ATTRIBUTE_COMPRESSED
The file or directory is compressed. For a file, this
means that all of the data in the file is compressed.
For a directory, this means that compression is the default 
for newly created files and subdirectories.
 
FILE_ATTRIBUTE_DEVICE
Reserved; do not use.
 
FILE_ATTRIBUTE_DIRECTORY
The handle identifies a directory.
 
FILE_ATTRIBUTE_ENCRYPTED
The file or directory is encrypted.
For a file, this means that all data in the file is encrypted.
For a directory, this means that encryption is the default for 
newly created files and subdirectories.
 
FILE_ATTRIBUTE_HIDDEN
The file or directory is hidden.
It is not included in an ordinary directory listing.
 
FILE_ATTRIBUTE_NORMAL
The file or directory does not have other attributes set.
This attribute is valid only when used alone.
 
FILE_ATTRIBUTE_NOT_CONTENT_INDEXED
The file is not to be indexed by the content indexing service.
 
FILE_ATTRIBUTE_OFFLINE
The file data is not available immediately.
This attribute indicates that the file data is physically moved to offline storage.
This attribute is used by Remote Storage, which is the hierarchical 
storage management software.
Note  Applications should not arbitrarily change this attribute.
 
FILE_ATTRIBUTE_READONLY
The file or directory is read-only.
For a file, applications can read the file, but cannot write to it or delete it.
For a directory, applications cannot delete it.
 
FILE_ATTRIBUTE_REPARSE_POINT
The file or directory has an associated reparse point.
 
FILE_ATTRIBUTE_SPARSE_FILE
The file is a sparse file.
 
FILE_ATTRIBUTE_SYSTEM
The file or directory is part of the operating system, or the operating 
system uses the file or directory exclusively.
 
FILE_ATTRIBUTE_TEMPORARY
The file is being used for temporary storage.
File systems attempt to keep all of the data in memory for quick 
access, rather than flushing it back to mass storage.
An application should delete a temporary file as soon as it is not needed.
 
FILE_ATTRIBUTE_VIRTUAL
The file is a virtual file.

ftCreationTime 
A FILETIME structure that specifies when a file or directory was created.

   ============================= */

// eof - winport.c


