// Font.cxx
// from : http://gpwiki.org/index.php/OpenGL:Tutorials:Font_System
#pragma warning(disable: 4267)

#include "ogl02.hxx"
#include "Font.hxx"
#include "TexFont.h"
#include <fstream>

static char * full_char_set = "abcdefghijklmnopqrstuvwxyz"
                          "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
                          "1234567890~!@#$%^&*()-=+;:"
                          "'\",./?[]|\\ <>`\xFF";

// Helper function to read a piece of data from a stream.
template<class T, class S>
void Read_Object(T& to_read, S& in)
{
  in.read(reinterpret_cast<char*>(&to_read), sizeof(T));
}
 
// This is how glyphs are stored in the file.
struct Glyph_Buffer
{
  unsigned char ascii, width;
  unsigned short x, y;
};
 
Font::Font(const char* filename)
  : _line_height(0),
    _texture(0),
    _tex_line_height(0),
    _valid_font(false),
    _is_tex_font(false),
    _txf(NULL),
    _glyphs(NULL)
{

  int width, height, n_chars, i;
  int c1, c2;
  unsigned char* tex_data = NULL;

  for (i = 0; i != 256; ++i)
    _table[i] = NULL;

  c1 = sizeof(int);
  c2 = sizeof(size_t);
  if( c1 != c2 ) {
      printf("NOTE: sizeof(int)=%d sizeof(size_t)=%d DIFFERENT!\n",
         c1, c2 );
      //fflush(stdout);
      //exit(1);
  }
  // Open the file and check whether it is any good (a font file
  // starts with "F0")
  std::ifstream input(filename, std::ios::binary);
  //  || input.get() != 'F' || input.get() != '0')
  if (input.fail())
    throw "Font file NOT available!";
    //throw std::runtime_error("Font file NOT available!");

  c1 = input.get();
  c2 = input.get();
  if(( c1 != 'F' )||( c2 != '0' )) {
     input.close();
     TexFont * ptf = txfLoadFont((char *)filename);
     if( !ptf )
         throw "NOT valid Font file (NOT 'FO', nor '0xFFtxf')!";
         //throw std::runtime_error("NOT valid Font file (NOT 'FO', nor '0xFFtxf')!");
      width   = ptf->tex_width;
      height  = ptf->tex_height;
      n_chars = ptf->num_glyphs;
      _line_height = ptf->max_height;
      _tex_line_height = static_cast<float>(_line_height) / height;
      tex_data = ptf->teximage;
      GLuint ret = txfEstablishTexture(ptf, _texture, false);
      _is_tex_font = true;
      _txf = ptf;
      return;
  }

  // Get the texture size, the number of glyphs and the line height.
  Read_Object(width, input);
  Read_Object(height, input);
  Read_Object(_line_height, input);
  Read_Object(n_chars, input);
  _tex_line_height = static_cast<float>(_line_height) / height;
 
#ifndef NDEBUG
   printf( "Got F0 txf width=%d, height=%d n_chars=%d\n",
      width, height, n_chars );
#endif
  // Make the glyph table.
  _glyphs = new Glyph[n_chars];
  // Read every glyph, store it in the glyph array and set the right
  // pointer in the table.
  Glyph_Buffer buffer;
  for (i = 0; i < n_chars; ++i){
    Read_Object(buffer, input);
    _glyphs[i].tex_x1 = static_cast<float>(buffer.x) / width;
    _glyphs[i].tex_x2 = static_cast<float>(buffer.x + buffer.width) / width;
    _glyphs[i].tex_y1 = static_cast<float>(buffer.y) / height;
    _glyphs[i].advance = buffer.width;
 
    _table[buffer.ascii] = _glyphs + i;
  }

  // Store the actual texture in an array.
  tex_data = new unsigned char[width * height];
  input.read(reinterpret_cast<char*>(tex_data), width * height);
 
  // All chars that do not have their own glyph are set to point to
  // the default glyph.
  Glyph* default_glyph = _table[(unsigned char)'\xFF'];
  // We must have the default character (stored under '\xFF')
  if (default_glyph == NULL)
    throw "Font file contains no default glyph";
    //throw std::runtime_error("Font file contains no default glyph");

  c1 = 0;
  for (i = 0; i != 256; ++i){
     if (_table[i] == NULL) {
        _table[i] = default_glyph;
        c1++;
     }
  }
#ifndef NDEBUG
  if(c1) {
      printf( "Filled in %d slots (of 256), with default glyph (NULL)\n", c1 );
  }
#endif // #ifndef NDEBUG

  // Generate an alpha texture with it.
  glGenTextures(1, &_texture);
  glBindTexture(GL_TEXTURE_2D, _texture);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexImage2D(GL_TEXTURE_2D, 0, GL_ALPHA8, width, height, 0, GL_ALPHA,
               GL_UNSIGNED_BYTE, tex_data);
  // And delete the texture memory block
  delete[] tex_data;
  _valid_font = true;
}

Font::~Font()
{
  // Release texture object and glyph array.
  glDeleteTextures(1, &_texture);
  if( _glyphs )
     delete[] _glyphs;
  if( _txf )
     txfUnloadFont(_txf);

}

int Font::Line_Height() const
{
  return _line_height;
}
int Font::Char_Width(unsigned char c) const
{
  return _table[c]->advance;
}
 
int Font::String_Width(const std::string& str) const
{
  int total = 0;
  for (int i = 0; i != (int)str.size(); ++i)
    total += Char_Width(str[i]);
  return total;
}

// This only leaves Draw_String. 
// This function just goes over every character in a string and 
// draws them next to each other. 
//
// It is left up to the caller to set the projection matrix to 
// orthogonal with 1 unit = 1 pixel, and to enable texturing and 
// blending. The text is drawn using OpenGL's current color. 
//
// If you want to show a piece of text on more than one line 
// you'll have to break it up into separate lines.

void Font::Draw_String(const std::string& str, float x, float y)
{
  glBindTexture(GL_TEXTURE_2D, _texture);
  if( _is_tex_font ) {
     txfRenderString(_txf, (char *)str.c_str(), str.size());
     return;
  }
 
  // Simply draw quads textured with the current glyph for every
  // character, updating the x position as we go along.
  glBegin(GL_QUADS);
  for (size_t i = 0; i != str.size(); ++i){
    Glyph* glyph = _table[str[i]];
 
    glTexCoord2f(glyph->tex_x1, glyph->tex_y1);
    glVertex2f(x, y);

    glTexCoord2f(glyph->tex_x1, glyph->tex_y1 + _tex_line_height);
    glVertex2f(x, y + _line_height);

    glTexCoord2f(glyph->tex_x2, glyph->tex_y1 + _tex_line_height);
    glVertex2f(x + glyph->advance, y + _line_height);

    glTexCoord2f(glyph->tex_x2, glyph->tex_y1);
    glVertex2f(x + glyph->advance, y);
 
    x += glyph->advance;
  }
  glEnd();
}

// eof - Font.cxx
