// ogl11.cxx
#include "ogl02.hxx"
// Example 11-4 : Quadrics Objects: quadric.c
// from : http://www.opengl.org/resources/code/samples/redbook/quadric.c

/*
 * Copyright (c) 1993-1997, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED 
 * Permission to use, copy, modify, and distribute this software for 
 * any purpose and without fee is hereby granted, provided that the above
 * copyright notice appear in all copies and that both the copyright notice
 * and this permission notice appear in supporting documentation, and that 
 * the name of Silicon Graphics, Inc. not be used in advertising
 * or publicity pertaining to distribution of the software without specific,
 * written prior permission. 
 *
 * THE MATERIAL EMBODIED ON THIS SOFTWARE IS PROVIDED TO YOU "AS-IS"
 * AND WITHOUT WARRANTY OF ANY KIND, EXPRESS, IMPLIED OR OTHERWISE,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY OR
 * FITNESS FOR A PARTICULAR PURPOSE.  IN NO EVENT SHALL SILICON
 * GRAPHICS, INC.  BE LIABLE TO YOU OR ANYONE ELSE FOR ANY DIRECT,
 * SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY
 * KIND, OR ANY DAMAGES WHATSOEVER, INCLUDING WITHOUT LIMITATION,
 * LOSS OF PROFIT, LOSS OF USE, SAVINGS OR REVENUE, OR THE CLAIMS OF
 * THIRD PARTIES, WHETHER OR NOT SILICON GRAPHICS, INC.  HAS BEEN
 * ADVISED OF THE POSSIBILITY OF SUCH LOSS, HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE
 * POSSESSION, USE OR PERFORMANCE OF THIS SOFTWARE.
 * 
 * US Government Users Restricted Rights 
 * Use, duplication, or disclosure by the Government is subject to
 * restrictions set forth in FAR 52.227.19(c)(2) or subparagraph
 * (c)(1)(ii) of the Rights in Technical Data and Computer Software
 * clause at DFARS 252.227-7013 and/or in similar or successor
 * clauses in the FAR or the DOD or NASA FAR Supplement.
 * Unpublished-- rights reserved under the copyright laws of the
 * United States.  Contractor/manufacturer is Silicon Graphics,
 * Inc., 2011 N.  Shoreline Blvd., Mountain View, CA 94039-7311.
 *
 * OpenGL(R) is a registered trademark of Silicon Graphics, Inc.
 */

/*
 *  quadric.c
 *  This program demonstrates the use of some of the gluQuadric*
 *  routines. Quadric objects are created with some quadric
 *  properties and the callback routine to handle errors.
 *  Note that the cylinder has no top or bottom and the circle
 *  has a hole in it.
 */

// 2009-06-18 - added to my OpenGL/GLUT example suite ogl02.exe
// Modified sample to use double buffering - avoid 'flickering' in WIN32,
// and some minor 'animation' - geoff mclane - geoffair at hotmail dot com
// =======================================================================
//#include <GL/gl.h>
//#include <GL/glu.h>
//#include <GL/glut.h>
#include <stdio.h>
//#include <stdlib.h>

static GLuint startList;
static GLfloat light_position[] = { 1.0, 1.0, 1.0, 0.0 };
static bool use_double_buffering = true;
static GLfloat cyl_angle = 300.0;
static GLfloat cyl_xyz[3] = { 1.0, 0.0, 0.0 };
static bool do_animation = true;
static GLfloat disk_angle = 0.0;
static GLenum sphere_model = GL_SMOOTH;
static GLenum cyl_model = GL_FLAT;
#ifdef _MSC_VER
#define STDCALL   __stdcall
#else
#define STDCALL GLAPIENTRY
#endif

static void STDCALL errorCallback(GLenum errorCode)
{
   const GLubyte *estring;
   estring = gluErrorString(errorCode);
   fprintf(stderr, "Quadric Error: %s\n", estring);
   pgm_exit(0);
}

static void set_light_pos(void)
{
   glLightfv(GL_LIGHT0, GL_POSITION, light_position);
}

static void move_light_pos(void)
{
   // light_position[] = { 1.0, 1.0, 1.0, 0.0 };
   light_position[0] = (light_position[0] > 0.0 ? -1.0 : 1.0);
   set_light_pos();
}

static void init114(void) 
{
   printf("Create 4 display lists, each with a different quadric object.\n");
   printf("Different drawing styles and surface normal specifications\n");
   printf("are demonstrated. ESC to exit.\n");

   GLUquadricObj *qobj;
   // The initial ambient reflectance for both
   // front- and back-facing materials is (0.2, 0.2, 0.2, 1.0).
   GLfloat mat_ambient[] = { 0.5, 0.5, 0.5, 1.0 };
   // The initial specular reflectance for both
   // front- and back-facing materials is (0, 0, 0, 1).
   GLfloat mat_specular[] = { 1.0, 1.0, 1.0, 1.0 };
   // The initial shininess exponent	for both front-	and
   // back-facing materials is 0.
   GLfloat mat_shininess[] = { 50.0 };
   //GLfloat light_position[] = { 1.0, 1.0, 1.0, 0.0 };
   GLfloat model_ambient[] = { 0.5, 0.5, 0.5, 1.0 };

   glClearColor(0.0, 0.0, 0.0, 0.0); // black background
   //glClearColor(0.2, 0.2, 0.2, 0.0); // gray background

   glMaterialfv(GL_FRONT, GL_AMBIENT, mat_ambient);
   glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
   glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);

   //glLightfv(GL_LIGHT0, GL_POSITION, light_position);
   set_light_pos();

   glLightModelfv(GL_LIGHT_MODEL_AMBIENT, model_ambient);

   glEnable(GL_LIGHTING);
   glEnable(GL_LIGHT0);
   glEnable(GL_DEPTH_TEST);

/* Create 4 display lists, each with a different quadric object.
 * Different drawing styles and surface normal specifications
 * are demonstrated.
 */
   startList = glGenLists(4); // get FOUR(4) lists
   qobj = gluNewQuadric();    // and Quadratic object
   CHKMEM(qobj);  // abort if memory allocation FAILED!
   gluQuadricCallback(qobj, GLU_ERROR, (void (STDCALL *)(void))errorCallback);

   printf("object 1 - sphere, filled, smooth shaded.\n");
   gluQuadricDrawStyle(qobj, GLU_FILL); /* smooth shaded */
   gluQuadricNormals(qobj, GLU_SMOOTH);
   glNewList(startList, GL_COMPILE);
      // Sphere(qobj,radius,slices,stacks)
      gluSphere(qobj, 0.75, 15,    10);
   glEndList();

   printf("object 2 - cylinder, filled, but flat shaded.\n");
   gluQuadricDrawStyle(qobj, GLU_FILL); /* flat shaded */
   gluQuadricNormals(qobj, GLU_FLAT);
   glNewList(startList+1, GL_COMPILE);
   // gluCylinder(*qobj, baseRadius, topRadius, height, slices, stacks)
      gluCylinder(qobj,  0.5,        0.3,       1.0,    15,     5);
   glEndList();

   printf("object 3 - disk, in wire frame form.\n");
   gluQuadricDrawStyle(qobj, GLU_LINE); /* wireframe */
   gluQuadricNormals(qobj, GLU_NONE);
   glNewList(startList+2, GL_COMPILE);
   // gluDisk(qobj,innerRadius,outerRadius,slices,loops)
      gluDisk(qobj, 0.25,      1.0,        20,    4);
   glEndList();

   printf("object 4 - partial disk, silhouette.\n");
   gluQuadricDrawStyle(qobj, GLU_SILHOUETTE);
   gluQuadricNormals(qobj, GLU_NONE);
   glNewList(startList+3, GL_COMPILE);
   // gluPartialDisk(qobj,innerRadius,outerRadius,slices,loops,startAngle,sweepAngle)
      gluPartialDisk(qobj, 0.0,       1.0,        20,    4,    0.0,       225.0);
   glEndList();

   gluDeleteQuadric(qobj); // be tidy
}

static void display114(void)
{
   glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

   glPushMatrix();

   glEnable(GL_LIGHTING);

   // object 1 - sphere - white
   //glShadeModel (GL_SMOOTH);
   glShadeModel (sphere_model);
   glTranslatef(-1.0, -1.0, 0.0);
   glCallList(startList);

   // object 2 - cylinder - white
   //glShadeModel (GL_FLAT);
   glShadeModel (cyl_model);
   glTranslatef(0.0, 2.0, 0.0);
   glPushMatrix();
      //glRotatef(300.0, 1.0, 0.0, 0.0); // from end-on view to side view
      glRotatef(cyl_angle, cyl_xyz[0], cyl_xyz[1], cyl_xyz[2] ); // rotate - angle,x,y,z
      glCallList(startList+1);
   glPopMatrix();

   glDisable(GL_LIGHTING);

   // object 3 - disk, in wire frame form
   glColor3f(0.0, 1.0, 1.0);  // blue/green
   glTranslatef(2.0, -2.0, 0.0); // x,y,x = move these to the right side, bottom -2
   glCallList(startList+2);

   // object 4 - partial disk
   glColor3f(1.0, 1.0, 0.0);  // yellow
   glTranslatef(0.0, 2.0, 0.0); // move to top 2
   glPushMatrix();
      glRotatef(disk_angle, 0.0, 0.0, 1.0 ); // rotate - angle,x,y,z
      glCallList(startList+3);
   glPopMatrix();

   glPopMatrix();

   if(use_double_buffering)
      glutSwapBuffers();
   else
      glFlush();

}

static void reshape114 (int w, int h)
{
   glViewport(0, 0, (GLsizei) w, (GLsizei) h);
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   // glOrtho( left, right, bottom, top,  nearVal, farVal ); 
   if (w <= h) {
      glOrtho(-2.5, 2.5,
         -2.5 *(GLfloat)h / (GLfloat)w, 2.5 * (GLfloat)h /(GLfloat)w,
         -10.0, 10.0);
   } else {
      glOrtho(-2.5 * (GLfloat)w / (GLfloat)h, 2.5 * (GLfloat)w /(GLfloat)h,
         -2.5, 2.5,
         -10.0, 10.0);
   }
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
}

static void key_help114(void)
{
   printf(" h/?   - this help.\n");
   printf(" a     - toggled animation (now %s)\n", do_animation ? "On" : "Off");
   printf(" r     - reset to defaults, and animation off\n");
   printf(" x     - toggle x rotation (now %s)\n", (cyl_xyz[0] > 0) ? "On" : "Off");
   printf(" y     - toggle y rotation (now %s)\n", (cyl_xyz[1] > 0) ? "On" : "Off");
   printf(" z     - toggle z rotation (now %s)\n", (cyl_xyz[2] > 0) ? "On" : "Off");
   printf(" s     - toggle sphere model (now %s)\n",
            (sphere_model == GL_SMOOTH) ? "GL_SMOOTH" : "GL_FLAT" );
   printf(" c     - toggle cylinder model (now %s)\n",
            (cyl_model == GL_SMOOTH) ? "GL_SMOOTH" : "GL_FLAT" );
   printf(" l     - toggle light position (now %s top)\n",
      (light_position[0] > 0.0) ? "right" : "left" );
   printf(" ESC/q - exit.\n" );
}

static void keyboard114(unsigned char key, int x, int y)
{
   keyboard_pgm_exit(key,x,y);
   switch (key) {
      case 'h':
      case '?':
         key_help114();
         break;
      case 'a':
         do_animation = !do_animation;
         printf("a - toggled animation to %s\n", do_animation ? "On" : "Off");
         break;
      case 'r':
         cyl_angle = 300.0;
         do_animation = false;
         printf("r - reset - animation to %s\n", do_animation ? "On" : "Off");
         cyl_xyz[0] = 1.0; // rotate only about x axis
         cyl_xyz[1] = 0.0;
         cyl_xyz[2] = 0.0;
         disk_angle = 0.0;
         sphere_model = GL_SMOOTH;
         cyl_model = GL_FLAT;
         light_position[0] = 1.0;
         glutPostRedisplay();
         break;
      case 'x':
         cyl_xyz[0] = (cyl_xyz[0] > 0) ? 0.0 : 1.0;
         printf("x - rotation to %s\n", (cyl_xyz[0] > 0) ? "On" : "Off");
         glutPostRedisplay();
         break;
      case 'y':
         cyl_xyz[1] = (cyl_xyz[1] > 0) ? 0.0 : 1.0;
         printf("y - rotation to %s\n", (cyl_xyz[1] > 0) ? "On" : "Off");
         glutPostRedisplay();
         break;
      case 'z':
         cyl_xyz[2] = (cyl_xyz[2] > 0) ? 0.0 : 1.0;
         printf("z - rotation to %s\n", (cyl_xyz[2] > 0) ? "On" : "Off");
         glutPostRedisplay();
         break;
      case 's':
         sphere_model = (sphere_model == GL_SMOOTH) ? GL_FLAT : GL_SMOOTH;
         printf("s - toggle sphere model to %s\n",
            (sphere_model == GL_SMOOTH) ? "GL_SMOOTH" : "GL_FLAT" );
         glutPostRedisplay();
         break;
      case 'c':
         cyl_model = (cyl_model == GL_SMOOTH) ? GL_FLAT : GL_SMOOTH;
         printf("c - toggle cylinder model to %s\n",
            (cyl_model == GL_SMOOTH) ? "GL_SMOOTH" : "GL_FLAT" );
         glutPostRedisplay();
         break;
      case 'l':
         move_light_pos();
         printf("l - toggled light position (now %s top)\n",
            (light_position[0] > 0.0) ? "right" : "left" );
         glutPostRedisplay();
         break;
   }
}

static void idle114(void)
{
   if(do_animation) {
      cyl_angle += 0.2;
      if(cyl_angle > 360.0) cyl_angle = 0.0;
      disk_angle += 0.2;
      if(disk_angle > 360.0) disk_angle = 0.0;
      glutPostRedisplay();
   }
}

//   case 31:
//      printf(" 31: main114(argc,argv); // four figures - sphere, cylinder, disk, partial disk.\n" );
int main114(int argc, char** argv)
{
   glutInit(&argc, argv);
   if(use_double_buffering)
      glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH);
   else
      glutInitDisplayMode(GLUT_SINGLE | GLUT_RGB | GLUT_DEPTH);
   glutInitWindowSize(500, 500); 
   glutInitWindowPosition(100, 100);
   glutCreateWindow(argv[0]);
   init114();
   glutDisplayFunc(display114); 
   glutReshapeFunc(reshape114);
   glutKeyboardFunc(keyboard114);
   glutIdleFunc(idle114);
   key_help114();
   glutMainLoop();
   return 0;
}

// eof -ogl11.cxx
