// polar.cxx
// from : http://www.sv.vt.edu/classes/opengl/demos/polarViewDemo.c
/*
 * Copyright (c) 1996-1999  Silicon Graphics, Inc.  All rights reserved.
 *
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL SILICON GRAPHICS BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT ADVISED OF THE
 * POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*  polarViewDemo.c - Show the effect of changes in the incidence and
 *      azimuth paramters in the polarView routine.
 *
 *  Left Mousebutton		- change azimuth angle
 *  Middle Mousebutton		- change incidence angle
 *  Right Mousebutton		- change the twist angle
 *  F1 key			- print help information
 *  R key                       - reset incidence, azimuth, and twist of eye
 *  Escape key                  - exit the program
 */

#include "ogl02.hxx"
//#include <GL/glut.h>		/* includes gl.h, glu.h */
#include <math.h>
#include <stdio.h>	/* for printf */
#include "axes.hxx"  // FOUND axes.c at http://www.sv.vt.edu/classes/opengl/lib/axes.c
#include "ogl-sprtf.hxx"
#include "ogl-utils.hxx"
#ifndef _MSC_VER
#include <string.h> // for strlen()
#endif

#ifdef _ogl_sprtf_hxx_
#undef   printf
#define  printf sprtf
#endif

// FEATURE
#define  ADD_ANIMATION

/*  Function Prototypes  */
static void  initgfx( void );
static void  drawScene( void );
static void  reshape( GLsizei, GLsizei );
static void  keyboard( GLubyte, GLint, GLint );
static void  specialkeys( GLint, GLint, GLint );
static void  mouse( GLint, GLint, GLint, GLint );
static void  motion( GLint, GLint );
static void resetView( void );
static void polarView( GLfloat, GLfloat, GLfloat, GLfloat);
static void printHelp( char * );
static void setDistance(void);

/* Global Definitions */
#define KEY_ESC	27	/* ascii value for the escape key */
#define DEF_NEAR_CLIP   	2.5
#define DEF_FAR_CLIP       7.0

/* Variables */
static char 		*progname;
static void 	  	*fixedFont;

static GLdouble		xStart = 0.0, yStart = 0.0;
static GLdouble		xMove = 0.0, yMove = 0.0;
static bool          validAction = false;

static GLfloat 		nearClip, farClip, distance, twistAngle, incAngle, azimAngle;

/* action values */
enum	actions { CHANGE_AZIM, CHANGE_INC, CHANGE_TWIST, CHANGE_NONE };
static GLint	curr_action = CHANGE_AZIM;
static GLint	prev_action = CHANGE_NONE;

int
main_polar( int argc, char *argv[] )
{
	GLsizei width, height;
	progname = get_file_name_ptr( argv[0] );

	glutInit( &argc, argv );
	width = glutGet( GLUT_SCREEN_WIDTH ); 
	height = glutGet( GLUT_SCREEN_HEIGHT );
	//glutInitWindowPosition( (width / 2) + 4, height / 4 );
	glutInitWindowPosition( 10.0, 10.0 );
	glutInitWindowSize( (width / 2) - 4, height / 2 );
	glutInitDisplayMode( GLUT_RGBA | GLUT_DEPTH | GLUT_DOUBLE );
	glutCreateWindow( "polarViewDemo" );
   //glutCreateWindow( argv[0] );
	initgfx();

	glutMouseFunc( mouse );
	glutMotionFunc( motion );
	glutKeyboardFunc( keyboard );
	glutSpecialFunc( specialkeys );
	glutReshapeFunc( reshape );
	glutDisplayFunc( drawScene ); 
   glutIdleFunc( NULL );   // not used to animate scene

	printHelp( progname );

	glutMainLoop();
   return 0;
}

// 	"Axes: X - red, Y - green, Z - blue\n\n"
static void
printHelp( char * progname )
{
   const char * help = "%s: combine modeling transformations to \n" 
	"view objects as though encased in a glass ball\n"
	"Axes: X - red, Y - green, Z - blue\n"
#ifdef ADD_ANIMATION
   "a key               - toggle animation\n"
#endif //#ifdef ADD_ANIMATION
	"Left Mousebutton    - change azimuth angle\n"
	"Middle Mousebutton  - change incidence angle\n"
	"Right Mousebutton   - change the twist angle\n" 
	"R key               - reset incidence, azimuth, and twist angles\n"
	"F1 Key              - print Help information\n"
	"Escape key          - exit the program\n";
	printf((char *)help, progname);
}

static void
initgfx( void )
{
	GLfloat   yellowAmbient[] = { 1.0f, 1.0f, 0.0f, 1.0f };
	GLfloat   yellowDiffuse[] = { 1.0f, 1.0f, 0.0f, 1.0f };
	GLfloat   whiteSpecular[] = { 1.0f, 1.0f, 1.0f, 1.0f };
	
	glClearColor( 0.0, 0.0, 0.0, 1.0 );
	glEnable( GL_DEPTH_TEST );

	/* Set up nearClip and farClip so that ( farClip - nearClip ) > maxObjectSize, */
	/* and determine the viewing distance (adjust for zooming) */
	nearClip = DEF_NEAR_CLIP;
	farClip  = DEF_FAR_CLIP;

	resetView();

	glMaterialfv( GL_FRONT, GL_AMBIENT, yellowAmbient );
	glMaterialfv( GL_FRONT, GL_DIFFUSE, yellowDiffuse );
	glMaterialfv( GL_FRONT, GL_SPECULAR, whiteSpecular );
	glMaterialf( GL_FRONT, GL_SHININESS, 100.0f );

	/* Turn on a default light */
	glEnable( GL_LIGHT0 );

	/* Set up fonts to use for rendering */
	fixedFont = GLUT_BITMAP_9_BY_15;
}

static int max_size;
static void
reshape( GLsizei width, GLsizei height )
{
	GLdouble			aspect;

   g_width = width;
   g_height = height;
   max_size = width;
   if(max_size > height)
      max_size = height;

	glViewport( 0, 0, width, height );

	aspect = (GLdouble) width / (GLdouble) height;

	glMatrixMode( GL_PROJECTION );
	glLoadIdentity();
	gluPerspective( 47.0, aspect, nearClip, farClip );
	glMatrixMode( GL_MODELVIEW );
}

static void 
keyboard( GLubyte key, GLint x, GLint y )
{
   keyboard_pgm_exit(key,x,y);
	switch (key) {
	case 'R':
	case 'r':
		resetView();
		glutPostRedisplay();
		break;
#ifdef ADD_ANIMATION
   case 'a':   //   "a key               - toggle animation\n"
      break;
#endif //#ifdef ADD_ANIMATION
	case KEY_ESC:	/* Exit whenever the Escape key is pressed */
		pgm_exit(0);
	}
}

static void 
specialkeys( GLint key, GLint x, GLint y )
{
	switch (key) {
	case GLUT_KEY_F1:	/* print Help */
		printHelp( progname );
		break;
	}
}

static void 
mouse( GLint button, GLint state, GLint x, GLint y )
{
	if (state == GLUT_DOWN) {
      validAction = false;
		switch (button) 
      {
		case GLUT_LEFT_BUTTON: 
         /* in case there are only two mouse
          * buttons, shift-left is equivalent to the
          * middle mouse button
          */
         if(curr_action != CHANGE_NONE)
            prev_action = curr_action;
         if (glutGetModifiers() == GLUT_ACTIVE_SHIFT)
            curr_action = CHANGE_INC;
         else
            curr_action = CHANGE_AZIM;
         validAction = true;
         break;
		case GLUT_MIDDLE_BUTTON: 
         if(curr_action != CHANGE_NONE)
            prev_action = curr_action;
			curr_action = CHANGE_INC;
         validAction = true;
			break;
		case GLUT_RIGHT_BUTTON: 
         if(curr_action != CHANGE_NONE)
            prev_action = curr_action;
			curr_action = CHANGE_TWIST;
         validAction = true;
			break;
      case 3:  // WM_MOUSEWHEEL (away)
         nearClip += 0.1;  // increasing moves object away
         farClip  += 0.1;  // but eventually it 'disappears'!
   		curr_action = CHANGE_NONE;
         setDistance();
#ifdef ADD_XTRA_DEBUG
         printf( "WM_MOUSEWHEEL:3:away: - near=%f, far=%f, dist=%f\n",
            nearClip, farClip, distance );
#endif // #ifdef ADD_XTRA_DEBUG
   		glutPostRedisplay();
         break;
      case 4:  // WM_MOUSEWHEEL (towards)
         nearClip -= 0.1;  // reducing brings object forward
         farClip  -= 0.1;  // moving offscreen
   		curr_action = CHANGE_NONE;
         setDistance();
#ifdef ADD_XTRA_DEBUG
         printf( "WM_MOUSEWHEEL:4:towards: - near=%f, far=%f, dist=%f\n",
            nearClip, farClip, distance );
#endif // #ifdef ADD_XTRA_DEBUG
   		glutPostRedisplay();
         break;
      default:
         const char * hlp = "Uncased event: button %d, state %d\n";
         printf( (char *)hlp, button, state );
   		curr_action = CHANGE_NONE;
         break;
		}

		/* Update the saved mouse position */
		xStart = x;
		yStart = y;
	} else {
      if(curr_action != CHANGE_NONE)
         prev_action = curr_action;
		curr_action = CHANGE_NONE;
	}
}

static bool validMove(GLint act)
{
   if((act == CHANGE_AZIM)||
      (act == CHANGE_INC)||
      (act == CHANGE_TWIST))
      return true;

   return false;
}

static void
motion( GLint x, GLint y )
{
   if(validMove(curr_action) && validAction) {
      xMove = (double)x - xStart;
      yMove = (double)y - yStart;
   }
	switch (curr_action) {
	case CHANGE_AZIM:
		/* Adjust the eye position based on the mouse position */
		azimAngle += (GLdouble) (x - xStart);
		break;
	case CHANGE_INC:
		/* Adjust the eye position based on the mouse position */
		incAngle -= (GLdouble) (y - yStart);
		break;
	case CHANGE_TWIST:
		/* Adjust the eye twist based on the mouse position */
		twistAngle = fmod(twistAngle+(x - xStart), 360.0);
		break;
   case CHANGE_NONE:
      break;
	default:
      const char * unk = "Unknown action %d\n";
		printf((char *)unk, curr_action);
	}
	
	/* Update the stored mouse position for later use */
	xStart = x;
	yStart = y;

	glutPostRedisplay();
}

static void setDistance(void)
{
	distance   = nearClip + (farClip - nearClip) / 2.0;
}

static void
resetView( void )
{
	nearClip   = DEF_NEAR_CLIP;
	farClip    = DEF_FAR_CLIP;
   setDistance();
	twistAngle = 0.0f;	/* rotation of viewing volume (camera) */
	incAngle   = 0.0f;
	azimAngle  = 0.0f;
}

static void
polarView( GLfloat distance, GLfloat azimuth, GLfloat incidence,
			GLfloat twist)
{
	glTranslatef( 0.0f, 0.0f, -distance);
	glRotatef( -twist, 0.0f, 0.0f, 1.0f);      // angle,x,y,z
	glRotatef( -incidence, 1.0f, 0.0f, 0.0f);  // angle,x,y,z
	glRotatef( -azimuth, 0.0f, 0.0f, 1.0f);    // angle,x,y,z
}

static void
drawNormal( GLfloat px, GLfloat py, GLfloat pz,
	GLfloat nx, GLfloat ny, GLfloat nz )
{
	glPushAttrib( GL_CURRENT_BIT | GL_LINE_BIT | GL_ENABLE_BIT );
	  glDisable( GL_LIGHTING );
	  glColor3f( 1.0f, 1.0f, 1.0f );
	  glLineWidth( 2.5 );
	  glPushMatrix();
	    glTranslatef( px, py, pz );
	    glBegin( GL_LINES );
	      glVertex3f( 0.0f, 0.0f, 0.0f );
	      glVertex3f( nx, ny, nz );
	    glEnd();
	  glPopMatrix();
	glPopAttrib();
} /* drawNormal */

static void
renderBitmapString( void *font, const char *string )
{
   int i;
   int len = (int) strlen(string);
   for (i = 0; i < len; i++) {
      glutBitmapCharacter(font, string[i]);
   }
}

static void
drawStatus( void )
{
	static char a[50];
	GLsizei winWidth, winHeight;
   GLfloat top = 15.0;
   //GLfloat skip = 20.0;
   GLfloat skip = glutBitmapHeight(fixedFont);
   GLfloat left = 10.0;
   GLfloat bottom = 15.0;

	/* save the ModelView matrix */
	glPushMatrix(); 

	/* clear any current modeling or viewing xforms */
	glLoadIdentity(); 


	/* save the Projection matrix */
	glMatrixMode( GL_PROJECTION );
	glPushMatrix();

		/* make new world space to display status in */
		glLoadIdentity();
		winWidth = glutGet(GLUT_WINDOW_WIDTH); 
		winHeight = glutGet(GLUT_WINDOW_HEIGHT);
		glOrtho( -0.5, winWidth-0.5, -0.5, winHeight-0.5, -1, 1.0 );

		glPushAttrib( GL_DEPTH_BUFFER_BIT );
	
			/* don't want text to be hidden */
			glDisable( GL_DEPTH_TEST ); 

			glColor3f( 1.0f, 1.0f, 0.0f );

			//glRasterPos3f( 10.0f, winHeight-15.0f, 0.0f );
			glRasterPos3f( 10.0f, winHeight-top, 0.0f );
			sprintf (a, "left mouse - change azimuth");
			renderBitmapString( fixedFont, a );

			//glRasterPos3f( 10.0f, winHeight-35.0f, 0.0f );
			glRasterPos3f( left, winHeight-(top + (skip * 1.0)), 0.0f );
			sprintf (a, "middle mouse - change incidence");
			renderBitmapString( fixedFont, a );

			//glRasterPos3f( 10.0f, winHeight-55.0f, 0.0f );
			glRasterPos3f( left, winHeight-(top + (skip * 2.0)), 0.0f );
			sprintf (a, "right mouse - change twist");
			renderBitmapString( fixedFont, a );

			//glRasterPos3f( 10.0f, winHeight-75.0f, 0.0f );
			glRasterPos3f( left, winHeight-(top + (skip * 3.0)), 0.0f );
			sprintf (a, "R key - reset view");
			renderBitmapString( fixedFont, a );

			/* draw at pixel coord 10, 65 */
			//glRasterPos3f( left, 65.0f, 0.0f ); 
			glRasterPos3f( left, bottom + (skip * 2), 0.0f ); 
			sprintf (a, "azimuth   is %4d", (GLint) azimAngle);
			renderBitmapString( fixedFont, a );

			//glRasterPos3f( left, 40.0f, 0.0f );
			glRasterPos3f( left, bottom + skip, 0.0f );
			sprintf (a, "incidence is %4d", (GLint) incAngle);
			renderBitmapString( fixedFont, a );

			//glRasterPos3f( left, 15.0f, 0.0f );
			glRasterPos3f( left, bottom, 0.0f );
			sprintf (a, "Twist     is %4d", (GLint) twistAngle);
			renderBitmapString( fixedFont, a );
	
		glPopAttrib();

	/* restore matrixes */
	glPopMatrix();

	glMatrixMode( GL_MODELVIEW );
	glPopMatrix();
} /* drawStatus */

static void xyzAxes(double length)
{
  glBegin(GL_LINES);
    glColor3d(1.0, 0.0, 0.0);    // red
    glVertex3d(0.0, 0.0, 0.0);      // center to
    glVertex3d(length, 0.0, 0.0);   // x-axis

    glColor3d(0.0, 1.0, 0.0);    // green
    glVertex3d(0.0, 0.0, 0.0);      // center to
    glVertex3d(0.0, length, 0.0);   // y-axis

    glColor3d(0.0, 0.0, 1.0);    // blue
    glVertex3d(0.0, 0.0, 0.0);      // center to
    glVertex3d(0.0, 0.0, length);   // z-axis
  glEnd();
}

static void
drawScene( void )
{
	GLfloat   lightPosition0[] = { 1.0f, 1.0f, 1.0f, 0.0f };

	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

	glPushMatrix();

	  glLightfv( GL_LIGHT0, GL_POSITION, lightPosition0 );

	  polarView( distance, azimAngle, incAngle, twistAngle );

	  /* Turn on lighting computations */
	  glEnable( GL_LIGHTING );
	
	  /* Draw a lit torus rotated 60 degrees about the x axis */
	  glPushMatrix();
	    glRotatef( 60.0f, 1.0f, 0.0f, 0.0f );  // angle,x,y,z
	    glutSolidTorus( 0.3, 0.6, 15, 15 ); // innerRad,outerRad,nsides,rings
	    /* Turn off lighting comptations */
	    glDisable( GL_LIGHTING );
	    /* Draw a line ( unlit ) representing the face normal */
	    drawNormal( 0.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f );
	  glPopMatrix();

     // Found a VERY SIMPLE -
     xyzAxes( (double)((double)max_size * 0.5) );

	  XYZaxes(); // EVENTUALLY FOUND THIS! with fat lines, and arrowheads at end - nice


	  glColor3f( 0.3f, 0.8f, 0.8f );
	  glutWireSphere( 1.5, 15, 15 );

	  /* label north and south poles - in RED */
	  glColor3f( 1.0f, 0.0f, 0.0f );
	  glRasterPos3f( 0.0f, 0.0f, -1.6f );
	  renderBitmapString( fixedFont, (const char *)"South Pole" );

	  /* glColor3f( 1.0f, 0.0f, 0.0f ); */
	  glRasterPos3f( 0.0f, 0.0f, 1.55f );
	  renderBitmapString( fixedFont, (const char *)"North Pole" );

	glPopMatrix();

	drawStatus();

	glutSwapBuffers();
}


/*
 * Compute lookup table of cos and sin values forming a cirle
 * Notes:
 *    It is the responsibility of the caller to free these tables
 *    The size of the table is (n+1) to form a connected loop
 *    The last entry is exactly the same as the first
 *    The sign of n can be flipped to get the reverse loop
 */
static void my_fghCircleTable(double **sint,double **cost,const int n)
{
   int i;
   /* Table size, the sign of n flips the circle direction */
   const int size = abs(n);
   /* Determine the angle between samples */
   const double angle = 2*M_PI/(double)( ( n == 0 ) ? 1 : n );
   /* Allocate memory for n samples, plus duplicate of first entry at the end */
   *sint = (double *) calloc(sizeof(double), size+1);
   *cost = (double *) calloc(sizeof(double), size+1);
   /* Bail out if memory allocation fails, fgError never returns */
   if (!(*sint) || !(*cost)) {
      const char * err = "ERROR: Failed to allocate memory in fghCircleTable";
      free(*sint);
      free(*cost);
      //fgError("Failed to allocate memory in fghCircleTable");
      printf((char *)err);
      pgm_exit(1);
   }

   /* Compute cos and sin around the circle */
   (*sint)[0] = 0.0;
   (*cost)[0] = 1.0;
   for ( i = 1; i < size; i++ ) {
      (*sint)[i] = sin(angle*i);
      (*cost)[i] = cos(angle*i);
   }
   /* Last sample is duplicate of the first */
   (*sint)[size] = (*sint)[0];
   (*cost)[size] = (*cost)[0];

}

/*
 * Draws a wire sphere
 */
void my_glutWireSphere(GLdouble radius, GLint slices, GLint stacks)
{
    int i,j;
    /* Adjust z and radius as stacks and slices are drawn. */
    double r;
    double x,y,z;

    /* Pre-computed circle */
    double *sint1,*cost1;
    double *sint2,*cost2;

    // FREEGLUT_EXIT_IF_NOT_INITIALISED ( "glutWireSphere" );
    my_fghCircleTable( &sint1, &cost1, -slices  );
    my_fghCircleTable( &sint2, &cost2,  stacks*2);

    /* Draw a line loop for each stack */
    for (i = 1; i < stacks; i++ ) {

       z = cost2[i];
       r = sint2[i];

       glBegin(GL_LINE_LOOP);
       for( j = 0; j <= slices; j++ ) {
          x = cost1[j];
          y = sint1[j];
          glNormal3d( x, y, z );
          glVertex3d( x*r*radius, y*r*radius, z*radius);
       }
       glEnd();
    }

    /* Draw a line loop for each slice */
    for ( i = 0; i < slices; i++ ) {

       glBegin(GL_LINE_STRIP);
       for( j = 0; j <= stacks; j++ ) {
          x = cost1[i]*sint2[j];
          y = sint1[i]*sint2[j];
          z = cost2[j];
          glNormal3d( x, y, z );
          glVertex3d( x*radius, y*radius, z*radius);
       }
       glEnd();

    }

    /* Release sin and cos tables */
    free(sint1);
    free(cost1);
    free(sint2);
    free(cost2);
}

// eof - polar.cxx
